/*
 * HID driver for the TimeLink TouchWin MultiTouch Panel, based on hid-multitouch.c.
 * Copyright (c) 2012 ShenZhen TimeLink Inc.
 *
 * This program is free software; You can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the Free
 * Software Foundation; Either version 2 of the License or (at your option) any
 * later version.
 */

#include <linux/version.h>
#include <linux/module.h>
#include <linux/device.h>
#include <linux/hid.h>
#include <linux/slab.h>
#include <linux/usb.h>
#if (LINUX_VERSION_CODE < KERNEL_VERSION(2,6,38))
#   include <linux/input.h>
#else
#   include <linux/input/mt.h>
#endif

#define NOT_SEEN_MEANS_UP

/* It is recommended to use MT protocol B than MT protocol A, because
 * MT protocol B is more effective. MT protocol B is available since kernel
 * version 2.6.36, but some APIs for it is added since 2.6.38 */
#   define USE_MT_PROTOCOL_A

#if (LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,38))
#   define HAS_FEATURE_MAPPING_SUPPORT
#endif

/* Device with quirk HID_QUIRK_MULTITOUCH might be skipped by hid-core 'cause
 * they thought that should be handled by hid-multitouch since v3.2. But
 * hid-multitouch won't accept our device until v3.5, because our device is not
 * in its device table. */
#if (LINUX_VERSION_CODE >= KERNEL_VERSION(3,2,0)) \
    && (LINUX_VERSION_CODE < KERNEL_VERSION(3,5,0))
#   define HIDE_MULTITOUCH_QUIRK
#endif

#define USB_VENDOR_ID_TIMELINK 0x2309
#define USB_DEVICE_ID_TIMELINK 0x1001
#define USB_DEVICE_ID_TIMELINK_V2 0x1005

#define REPORTID_DEFAULT        0x00  // Not used.
#define REPORTID_MTOUCH         0x01
#define REPORTID_FEATURE        0x02
#define REPORTID_MOUSE          0x03
#define REPORTID_VENDOR_01      0x04
#define REPORTID_KEYBOARD       0x05
#define REPORTID_STOUCH         0x06
#define REPORTID_VENDOR_DRIVER  0xFF
#define SUBID_VD_MTTOUCH        0x01


#define timelink_error(fmt, ...) \
    printk(KERN_ERR "hid-timelink: "fmt"\n", ##__VA_ARGS__)

#define timelink_info(fmt, ...) \
    printk(KERN_INFO "hid-timelink: "fmt"\n", ##__VA_ARGS__)

#define timelink_debug(fmt, ...) \
    printk(KERN_DEBUG "hid-timelink: "fmt"\n", ##__VA_ARGS__)


MODULE_AUTHOR("Canmor Lam < canmor.lam@gmail.com >");
MODULE_DESCRIPTION ("TimeLink TouchWin Multitouch Panel");
MODULE_LICENSE("GPL");


typedef struct timelink_contact {
    bool state;
    unsigned char id;
    unsigned short x;
    unsigned short y;
    unsigned short width;
    unsigned short height;
    unsigned char p;
} contact_t;

typedef struct timelink_slot {
    contact_t contact; 
    bool state;
} slot_t;

typedef struct timelink_feature {
    int x_mininum;
    int x_maxinum;
    int y_mininum;
    int y_maxinum;
    int width_mininum;
    int width_maxinum;
    int height_mininum;
    int height_maxinum;
} feature_t;

typedef struct timelink_device {
    struct hid_input* hidinput;
    __u8 int_in_addr;
    __u8 int_out_addr;
    slot_t* slots;
    slot_t incoming_slot;
    slot_t* received_slots;
    size_t max_contact_count;
    size_t received_contact_count;
    size_t expected_contact_count;
    size_t last_field_index;
    size_t last_slot_field;
    int last_mt_collection;
    bool idle;
    int pointer_index;
    feature_t feature;
} device_t;


static __u8* timelink_report_fixup(struct hid_device* hdev, __u8* rdesc,
        unsigned int* rsize)
{
    /* Currently firmware report was supported,
     * so we won't be necessary to fix the report. */
    return rdesc;
}

static int timelink_input_mapping(struct hid_device* hdev, 
                struct hid_input* hi, struct hid_field* field,
                struct hid_usage* usage, unsigned long** bit, int* max)
{
    device_t* device = hid_get_drvdata(hdev);

    /* Only map fields from Touchscreen collections.
     * We need to ignore fields that belong to other collections
     * such as Mouse that might have the same Generic Desktop usages. */
    if (field->application == HID_DG_TOUCHSCREEN)
#ifdef INPUT_PROP_DIRECT
        set_bit(INPUT_PROP_DIRECT, hi->input->propbit);
#else
        ;
#endif
    else
        return -1;

    switch (usage->hid & HID_USAGE_PAGE)
    {
    case HID_UP_GENDESK:
        switch (usage->hid)
        {
        case HID_GD_X:
            device->feature.x_mininum = field->logical_minimum;
            device->feature.x_maxinum = field->logical_maximum;
            hid_map_usage(hi, usage, bit, max, EV_ABS, ABS_MT_POSITION_X);
            input_set_abs_params(hi->input, ABS_MT_POSITION_X, 
                            device->feature.x_mininum,
                            device->feature.x_maxinum, 0, 0);
            input_set_abs_params(hi->input, ABS_X, 
                            device->feature.x_mininum,
                            device->feature.x_maxinum, 0, 0);
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;

        case HID_GD_Y:
            device->feature.y_mininum = field->logical_minimum;
            device->feature.y_maxinum = field->logical_maximum;
            hid_map_usage(hi, usage, bit, max, EV_ABS, ABS_MT_POSITION_Y); 
            input_set_abs_params(hi->input, ABS_MT_POSITION_Y, 
                            device->feature.y_mininum,
                            device->feature.y_maxinum, 0, 0);
            input_set_abs_params(hi->input, ABS_Y,
                            device->feature.y_mininum,
                            device->feature.y_maxinum, 0, 0);
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;
        }
        return 0;

    case HID_UP_DIGITIZER:
        switch (usage->hid)
        {
        case HID_DG_TIPSWITCH:
            hid_map_usage(hi, usage, bit, max, EV_KEY, BTN_TOUCH);
            input_set_capability(hi->input, EV_KEY, BTN_TOUCH);
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;
        case HID_DG_CONFIDENCE:
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;
        case HID_DG_WIDTH:
            device->feature.width_mininum = field->logical_minimum;
            device->feature.width_maxinum = field->logical_maximum;
            hid_map_usage(hi, usage, bit, max, EV_ABS, ABS_MT_TOUCH_MAJOR);
            input_set_abs_params(hi->input, ABS_MT_TOUCH_MAJOR,
                            device->feature.width_mininum,
                            device->feature.width_maxinum, 0, 0);
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;
        case HID_DG_HEIGHT:
            device->feature.height_mininum = field->logical_minimum;
            device->feature.height_maxinum = field->logical_maximum;
            hid_map_usage(hi, usage, bit, max, EV_ABS, ABS_MT_TOUCH_MINOR);
            input_set_abs_params(hi->input, ABS_MT_TOUCH_MINOR,
                            device->feature.height_mininum,
                            device->feature.height_maxinum, 0, 0);
            input_set_abs_params(hi->input, ABS_MT_ORIENTATION, 0, 1, 0, 0);
            if (device->last_mt_collection == usage->collection_index) {
                device->last_slot_field = usage->hid;
                device->last_field_index = field->index;
            }
            return 1;
        case HID_DG_CONTACTID:
#ifndef HAS_FEATURE_MAPPING_SUPPORT
            device->max_contact_count = 48;
#endif
            if (device->max_contact_count > 0) {
#ifdef USE_MT_PROTOCOL_B
                input_mt_init_slots(hi->input, device->max_contact_count);
#else
                hid_map_usage(hi, usage, bit, max, EV_ABS, ABS_MT_TRACKING_ID);
                input_set_abs_params(hi->input, ABS_MT_TRACKING_ID, 
                                field->logical_minimum,
                                field->logical_maximum, 0, 0);
#endif
                device->hidinput = hi;
            }
            device->last_mt_collection = usage->collection_index;
            return 1;
        case HID_DG_CONTACTCOUNT:
            /* in the report descriptor of our device, contact max are global,
             * therefore, collection_index shouldn't be test */
            device->last_field_index = field->index;
            return 1;
        }
        return 0;

    case 0xff000000:
        /* Ignore HID features; */
        return -1;
    }
    return -1;
}

static int timelink_input_mapped(struct hid_device* hdev, 
                struct hid_input* hi, struct hid_field* field,
                struct hid_usage* usage, unsigned long** bit, int* max)
{
    if (usage->type == EV_KEY || usage->type == EV_ABS)
        set_bit(usage->type, hi->input->evbit);
    return -1;
}

#ifdef HAS_FEATURE_MAPPING_SUPPORT
static void timelink_feature_mapping(struct hid_device* hdev,
            struct hid_field* field, struct hid_usage* usage)
{
    device_t* device = hid_get_drvdata(hdev);
    switch (usage->hid)
    {
    case HID_DG_CONTACTMAX:
        /* Our device always report this value to be 0, so I hard-code it.  */
        device->max_contact_count = 48;
        break;
    }
}
#endif

static void timelink_complete_slot(device_t* td)
{
    if (td->received_slots == NULL)
        return;

    if (td->incoming_slot.state) {
        int slotnum = td->received_contact_count;
        if (slotnum >= 0 && slotnum < td->max_contact_count) {
            td->received_slots[slotnum] = td->incoming_slot;
        }
    }
}

static void timelink_complete_slots(device_t* device)
{
    size_t i, count = device->received_contact_count;

    if (device->received_slots == NULL || device->slots == NULL)
        return;

    /* Some legacy data might be received with the firmwares before v4.2,
     * this is going to fix that. */
    count = min(count, device->expected_contact_count);

    for (i = 0; i < count; ++i) {
        size_t slotnum = device->received_slots[i].contact.id % device->max_contact_count;
        device->slots[slotnum] = device->received_slots[i];
    }

}

static void timelink_clean_slots(device_t* device)
{
    size_t i;

    if (device->slots == NULL)
        return;

    for (i = 0; i < device->max_contact_count; ++i) {
        device->slots[i].state = false;
#ifdef NOT_SEEN_MEANS_UP
        device->slots[i].contact.state = false;
#endif
    }
}

#ifndef USE_MT_PROTOCOL_B
static int timelink_find_first_active_slot(const slot_t* slots, size_t count)
{
    int i;

    for (i = 0; i < count; ++i) {
        if (slots[i].state && slots[i].contact.state)
            return i;
    }
    return -1;
};

static void timelink_pointer_emulation(device_t* device, struct input_dev* input)
{
    int i = timelink_find_first_active_slot(device->slots, device->max_contact_count);
    if (i >= 0) {
        if (device->idle) {
            input_event(input, EV_KEY, BTN_TOUCH, 1);
            device->idle = false;
            device->pointer_index = i;
        }
        if (device->slots[device->pointer_index].state
            && device->slots[device->pointer_index].contact.state) {
            input_event(input, EV_ABS, ABS_X, device->slots[device->pointer_index].contact.x);
            input_event(input, EV_ABS, ABS_Y, device->slots[device->pointer_index].contact.y);
        }
    }
    else {
        if (!device->idle) {
            input_event(input, EV_KEY, BTN_TOUCH, 0);
            device->idle = true;
            device->pointer_index = -1;
        }
    }
}

static void timelink_emit_event_a(device_t* device, struct input_dev* input)
{
    size_t i;
    int orient, major, minor;

    for (i = 0; i < device->max_contact_count; ++i) {
        slot_t* s = device->slots + i;

        if (!s->state)
            continue;

        /* In order to avoid reporting wrong details of disappearing contact,
         * details will be ignored. */
        if (s->contact.state) {
            orient = (s->contact.width > s->contact.height);
            major = max(s->contact.width, s->contact.height);
            minor = min(s->contact.width, s->contact.height);

            input_event(input, EV_ABS, ABS_MT_TRACKING_ID, s->contact.id);
            input_event(input, EV_ABS, ABS_MT_POSITION_X, s->contact.x);
            input_event(input, EV_ABS, ABS_MT_POSITION_Y, s->contact.y);
            input_event(input, EV_ABS, ABS_MT_TOUCH_MAJOR, major);
            input_event(input, EV_ABS, ABS_MT_TOUCH_MINOR, minor);
            input_event(input, EV_ABS, ABS_MT_ORIENTATION, orient);
            input_mt_sync(input);
        }
    }
    timelink_pointer_emulation(device, input);
    input_sync(input);
}

#else /* #ifndef USE_MT_PROTOCOL_B */

static void timelink_emit_event_b(device_t* device, struct input_dev* input)
{
    size_t i;
    int orient, major, minor;

    for (i = 0; i < device->max_contact_count; ++i) {
        slot_t* s = device->slots + i;

        input_mt_slot(input, i);
        input_mt_report_slot_state(input, MT_TOOL_FINGER, s->contact.state);
        /* In order to avoid reporting wrong details of disappearing contact,
         * details will be ignore. */
        if (s->contact.state) {
            orient = (s->contact.width > s->contact.height);
            major = max(s->contact.width, s->contact.height);
            minor = min(s->contact.width, s->contact.height);

            input_event(input, EV_ABS, ABS_MT_POSITION_X, s->contact.x);
            input_event(input, EV_ABS, ABS_MT_POSITION_Y, s->contact.y);
            input_event(input, EV_ABS, ABS_MT_TOUCH_MAJOR, major);
            input_event(input, EV_ABS, ABS_MT_TOUCH_MINOR, minor);
            input_event(input, EV_ABS, ABS_MT_ORIENTATION, orient);
        }
    }
    input_mt_report_pointer_emulation(input, true);
    input_sync(input);
}
#endif

static void timelink_emit_event(device_t* device, struct input_dev* input)
{
    if (input == NULL || device->slots == NULL)
        return;

#ifdef USE_MT_PROTOCOL_B
    timelink_emit_event_b(device, input);
#else
    timelink_emit_event_a(device, input);
#endif
}

static int timelink_events(struct hid_device* hid,
            struct hid_field* field,
            struct hid_usage* usage,
            __s32 value)
{
    device_t *device = hid_get_drvdata(hid);

    if (hid->claimed & HID_CLAIMED_INPUT)
    {
        switch (usage->hid) {
        case HID_DG_TIPSWITCH:
            device->incoming_slot.contact.state = value;
            break;
        case HID_DG_CONFIDENCE:
            device->incoming_slot.state = value;
            break;
        case HID_DG_CONTACTID:
            device->incoming_slot.contact.id = value;
            break;
        case HID_DG_TIPPRESSURE:
            device->incoming_slot.contact.p = value;
            break;
        case HID_GD_X:
            device->incoming_slot.contact.x = value;
            break;
        case HID_GD_Y:
            device->incoming_slot.contact.y = value;
            break;
        case HID_DG_WIDTH:
            device->incoming_slot.contact.width = value;
            break;
        case HID_DG_HEIGHT:
            device->incoming_slot.contact.height = value;
            break;
        case HID_DG_CONTACTCOUNT:
            /* Includes multi-packet support where subsequent
             * packets are sent with zero of contact count. */
            if (value)
                device->expected_contact_count = value;
            break;
        case 0xff000002:
            /* Actually, the Vendor defined usage report will be here.
             * This report contains the data of sensors, that will be used to
             * compute the contacts in user space.
             * Therefore, just fallback to the generic hidinput handling. */
            // fall-through!
        default:
            /* fallback to the generic hidinput handling */
            return 0;
        }

        if (usage->hid == device->last_slot_field) {
            timelink_complete_slot(device);
            ++device->received_contact_count;
        }

        if (field->index == device->last_field_index) {
            if (device->received_contact_count >= device->expected_contact_count) {
                timelink_complete_slots(device);
                timelink_emit_event(device, device->hidinput->input);
                timelink_clean_slots(device);
                device->received_contact_count = 0;
                device->expected_contact_count = 0;
            }
        }

    }

    /* We have handled the hidinput part, now the hiddev */
    if (hid->claimed & HID_CLAIMED_HIDDEV && hid->hiddev_hid_event)
        hid->hiddev_hid_event(hid, field, usage, value);

    return 1;
}

/*
 * Data format:
 * 0               8               16              24              32
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | status        | id            | x                             |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | y                             | width                         |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | height                        | ...
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
static bool timelink_parse_contact(contact_t* contact, const __u8* buf, size_t size)
{
    if (!buf || size < 10)
        return false;

    contact->state = true;
    contact->id = *(buf + 1);
    contact->x = le16_to_cpup((__le16*)(buf + 2));
    contact->y = le16_to_cpup((__le16*)(buf + 4));
    contact->width = le16_to_cpup((__le16*)(buf + 6));
    contact->height = le16_to_cpup((__le16*)(buf + 8));
    /* no pressure support from device yet */
    contact->p = 0;
    return true;
};

static void timelink_transform_contact(const feature_t* feature, contact_t* contact)
{
    const int dx = feature->x_maxinum - feature->x_mininum;
    const int dy = feature->y_maxinum - feature->y_mininum;
    const int dw = feature->width_maxinum - feature->width_mininum;
    const int dh = feature->height_maxinum - feature->height_mininum;

    contact->x = feature->x_mininum + contact->x * dx / 0x7FFF;
    contact->y = feature->y_mininum + contact->y * dy / 0x7FFF;
    contact->width = feature->width_mininum + contact->width * dw / 0x7FFF;
    contact->height = feature->height_mininum + contact->height * dh / 0x7FFF;
};

/*
 * Data format:
 * 0               8               16              24              32
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | size          | count         | rest          | data(s) ...   
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 */
static int timelink_parse_output_contacts_report(device_t* device, __u8* buf, size_t size)
{
    size_t result = 0;
    size_t data_size, data_count, rest_data_count;
    bool report_complete = false, necessary_to_report = false;

    if (!buf || size < 3)
        return result;

    data_size = buf[0];
    data_count = buf[1];
    data_count = (data_size == 0) ? 0 : min(data_count, (size - 3) / data_size);
    rest_data_count = buf[2];
    
    if (device->received_contact_count == 0 && device->expected_contact_count == 0) {
        device->expected_contact_count = data_count + rest_data_count;
    }

    for (result = 0; result < data_count; ++result) {
        timelink_parse_contact(&device->incoming_slot.contact, buf + 3 + data_size * result, data_size);
        timelink_transform_contact(&device->feature, &device->incoming_slot.contact);
        device->incoming_slot.state = true;
        timelink_complete_slot(device);
        ++device->received_contact_count;
    }

    report_complete = device->received_contact_count == device->expected_contact_count;
    necessary_to_report = device->received_contact_count > 0 || !device->idle;
    if (report_complete && necessary_to_report) {
        if (device->hidinput) {
            timelink_complete_slots(device);
            timelink_emit_event(device, device->hidinput->input);
            timelink_clean_slots(device);
        }
        device->idle = (device->received_contact_count == 0);
        device->received_contact_count = 0;
        device->expected_contact_count = 0;
    }

    return result;
}

/*
 * The TimeLink TouchWin does not handle HID Output Reports
 * like it should according to usbhidm /hid-core.c::usbhid_output_raw_report()
 * so we need to override it.
 *
 * Data format:
 * 0               8               16              24              32 
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | report id     | sub-id        | count | index | state | def   |
 * +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
 * | checksum      | data(s) ...      
 * +-+-+-+-+-+-+-+-+-+-+-+-+-
 */
static int timelink_output_raw_report(struct hid_device *hid, __u8 *buf,
        size_t count, unsigned char report_type)
{
    device_t* device = hid_get_drvdata(hid);
    struct usb_interface *intf = to_usb_interface(hid->dev.parent);
    struct usb_device *dev = interface_to_usbdev(intf);
    int report_id;
    int sub_id;
    int actual_sent = 0;
    int ret = 0;

    if (!buf || count < 5)
        return ret;

    report_id = buf[0];
    sub_id = buf[1];

    if (report_id == REPORTID_VENDOR_DRIVER) {    
        if (sub_id == SUBID_VD_MTTOUCH) {
            timelink_parse_output_contacts_report(device, buf + 5, count - 5);
            ret = count;
        }
    }
    else {
        // FIXME: instead of hard coding output address.
        ret = usb_interrupt_msg(dev, usb_sndintpipe(dev, 1),
            buf, count, &actual_sent,
            USB_CTRL_SET_TIMEOUT);
        /* Count also the Report ID, in case of an Output report. */
        if (ret == 0) {
            ret = actual_sent;
        }
    }

    return ret;
}

static int timelink_probe(struct hid_device* hdev, const struct hid_device_id* id)
{
    int ret;
#ifdef HIDE_MULTITOUCH_QUIRK
    bool has_multitouch_quirk_hided = false;
#endif

    device_t* device = NULL;
    device = kzalloc(sizeof(device_t), GFP_KERNEL);
    if (!device)
    {
        timelink_error("Cannot allocate data.");
        return -ENOMEM;
    }
    hid_set_drvdata(hdev, device);

    ret = hid_parse(hdev);
    if (ret)
    {
        timelink_error("Cannot parse device.");
        goto err_free_device;
    }

#ifdef HIDE_MULTITOUCH_QUIRK
    /* Device with quirk HID_QUIRK_MULTITOUCH may be skipped by hid-core 'cause
     * they think that should be handled by hid-multitouch.
     * Therefore, I remove HID_QUIRK_MULTITOUCH before calling hid_hw_start. */
    if (hdev->quirks & HID_QUIRK_MULTITOUCH) {
        hdev->quirks &= ~HID_QUIRK_MULTITOUCH;
        has_multitouch_quirk_hided = true;
    }
#endif

    ret = hid_hw_start(hdev, HID_CONNECT_DEFAULT);
    if (ret)
    {
        timelink_error("Cannot start device.");
        goto err_free_device;
    }

#ifdef HIDE_MULTITOUCH_QUIRK
    /* Correspondingly, put quirk HID_QUIRK_MULTITOUCH back. */
    if (has_multitouch_quirk_hided) {
        hdev->quirks |= HID_QUIRK_MULTITOUCH;
        has_multitouch_quirk_hided = false;
    }
#endif

    device->received_slots = kzalloc(sizeof(slot_t) * device->max_contact_count, GFP_KERNEL);
    if (!device->received_slots) 
    {
        timelink_error("Cannot allocate multitouch received slots");
        ret = -ENOMEM;
        goto err_free_received_slots;
    }
    device->slots = kzalloc(sizeof(slot_t) * device->max_contact_count, GFP_KERNEL);
    if (!device->slots) 
    {
        timelink_error("Cannot allocate multitouch slots");
        ret = -ENOMEM;
        goto err_stop_hw;
    }
    device->idle = true;

    hdev->hid_output_raw_report = timelink_output_raw_report;
    
    struct usb_device *dev = container_of(hdev->dev.parent->parent, struct usb_device, dev);
    int actual_sent = 0;
    char buf[64] = {0x04,0x20,0x10,0x00,0x00,0x02};
    char buf2[3] = {0x02, 0x02, 0x00};
    ret = usb_interrupt_msg(dev, usb_sndintpipe(dev, 1),
                            buf2, sizeof(buf2), &actual_sent,
                            USB_CTRL_SET_TIMEOUT);
    ret = usb_interrupt_msg(dev, usb_sndintpipe(dev, 1),
                            buf, sizeof(buf), &actual_sent,
                            USB_CTRL_SET_TIMEOUT);

    return 0;

err_free_received_slots:
    kfree(device->received_slots);
err_stop_hw:
    hid_hw_stop(hdev);
err_free_device:
    kfree(device);
return ret;
}

static void timelink_removed(struct hid_device* hdev)
{
    device_t* device = hid_get_drvdata(hdev);
    hid_hw_stop(hdev);
    kfree(device->received_slots);
    kfree(device->slots);
    kfree(device);
    hid_set_drvdata(hdev, NULL);
}

static const struct 
hid_device_id timelink_devices[] =
{
    { HID_USB_DEVICE(USB_VENDOR_ID_TIMELINK, USB_DEVICE_ID_TIMELINK) } ,
    { HID_USB_DEVICE(USB_VENDOR_ID_TIMELINK, USB_DEVICE_ID_TIMELINK_V2) } ,
    {}
};

MODULE_DEVICE_TABLE(hid, timelink_devices);

static const struct hid_usage_id timelink_grabbed_usages[] = 
{
    { HID_ANY_ID, HID_ANY_ID, HID_ANY_ID } ,
    { HID_ANY_ID - 1, HID_ANY_ID - 1, HID_ANY_ID - 1 }
};

static struct hid_driver timelink_driver =
{
    .name          = "hid-timelink",
    .id_table      = timelink_devices,
    .probe         = timelink_probe,
    .remove        = timelink_removed,
    .input_mapping    = timelink_input_mapping,
    .input_mapped     = timelink_input_mapped,
#ifdef HAS_FEATURE_MAPPING_SUPPORT
    .feature_mapping  = timelink_feature_mapping,
#endif
    .usage_table      = timelink_grabbed_usages,
    .event            = timelink_events,
    .report_fixup     = timelink_report_fixup,
};

static int __init timelink_init(void)
{
    return hid_register_driver(&timelink_driver);
}

static void __exit timelink_exit(void)
{
    hid_unregister_driver(&timelink_driver);
}

module_init(timelink_init);
module_exit(timelink_exit);

